/*
 * Copyright (c) 2010, Nieko Maatjes
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. This program may not be used for commercial purposes without specific
 *    prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY NIEKO MAATJES ''AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL NIEKO MAATJES BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "main.h"

using namespace std;

#define MAX(a, b) ((a) > (b) ? (a) : (b))
#define MIN(a, b) ((a) < (b) ? (a) : (b))
#define SQUARE(a) ((a)*(a))

PSP_MODULE_INFO("PSPlanets", PSP_MODULE_USER, 0, 1); // version: major, minor
PSP_MAIN_THREAD_ATTR(PSP_THREAD_ATTR_USER | PSP_THREAD_ATTR_VFPU);
PSP_HEAP_SIZE_KB(20000); // 20 MB for malloc()

enum state_t { done, addPlanet, setVector };
enum state_t currentState;

// CTRL
SceCtrlData pad, oldPad;

// EXTERN
extern unsigned int __attribute__((aligned(16))) guList[262144];

int main()
{
    setupExitCallback();
    graphicsInit();

    currentState = addPlanet;

    sceCtrlSetSamplingMode(PSP_CTRL_MODE_ANALOG);

    while (1)
    {
        oldPad.Buttons = pad.Buttons;
        sceCtrlReadBufferPositive(&pad, 1);

        // User pressed X, different state
        if ((pad.Buttons ^ oldPad.Buttons)
         && (pad.Buttons & PSP_CTRL_CROSS))
        {
            switch (currentState)
            {
                default:
                case done:
                    currentState = addPlanet;
                    break;
                case addPlanet:
                    currentState = setVector;
                    break;
                case setVector:
                    // Finish and launch last planet
                    Planet::planets.back()->finish();
                    currentState = done;
                    break;
            }
        }
        // Reset
        else if (currentState == done
              && (pad.Buttons ^ oldPad.Buttons)
              && (pad.Buttons & PSP_CTRL_TRIANGLE))
        {
            Planet::clearAll();
            currentState = addPlanet;
        }
        // Switch to a demo
        else if ((pad.Buttons ^ oldPad.Buttons)
              && (pad.Buttons & PSP_CTRL_SQUARE))
        {
            Planet::clearAll();
            Planet::planets.push_back(new Planet(1e4, 240, 136));
            Planet::planets.push_back(new Planet(1e0, 120, 136, 9.12));
            Planet::planets.push_back(new Planet(1e0, 320, 136, 11.15, GU_PI));
            Planet::planets.push_back(new Planet(1e0, 240,  96, 15.50, GU_PI*1.5));
            Planet::planets.push_back(new Planet(1e0, 180, 136, 12.84));
            Planet::planets.push_back(new Planet(1e2,  40, 136, 7.07));
            Planet::planets.push_back(new Planet(1e0,  32, 136, 10.52));
            Planet::finishAll();
        }

        // Start graphics
        sceGuStart(GU_DIRECT, guList);
        sceGuClearColor(0);
        sceGuClear(GU_COLOR_BUFFER_BIT);

        switch (currentState)
        {
            default:
            case done:
                movePlanets();
                drawPlanets();
                break;
            case addPlanet:
                createPlanet();
                drawPlanets();
                break;
            case setVector:
                directPlanet();
                drawPlanets();
                break;
        }

        graphicsFlush();
        sceDisplayWaitVblankStart();
        //sceKernelDelayThread(3e5);
    }

    die(NULL);

    return 0;
}

/*
 * Any planet has its own momentum, kg*m/s,
 * and during any second a number of forces,
 * kg*m/s^2. The new direction and speed are
 * determined by the sum of these vectors
 * of momentum and force over a second.
 */
void movePlanets()
{
    float centerX = 0, centerY = 0;
    for (unsigned int i = 0; i < Planet::planets.size(); i++)
    {
        // Add planet's own momentum (0° is north, 90° is east)
        float x = sinf(Planet::planets[i]->getDirection())*Planet::planets[i]->getSpeed();
        float y = cosf(Planet::planets[i]->getDirection())*Planet::planets[i]->getSpeed();

        // Add gravitational forces from other planets
        for (unsigned int j = 0; j < Planet::planets.size(); j++)
        {
            if (i == j) { continue; }

            // TODO: what to do if planets overlap within one pixel (abs(float a - float b) < 1)
            // TODO: make them crash into each other? :)

            float diffX = Planet::planets[j]->getX() - Planet::planets[i]->getX();
            float diffY = Planet::planets[j]->getY() - Planet::planets[i]->getY();
            float distance = sqrtf(SQUARE(diffX) + SQUARE(diffY));

            float force = (Planet::planets[i]->getMass() * Planet::planets[j]->getMass())
                         /(SQUARE(distance));

            // Change vector to represent force (change
            // in momentum) on a planet, not distance
            diffX *= force/Planet::planets[i]->getMass()/distance;
            diffY *= force/Planet::planets[i]->getMass()/distance;

            // Add this vector
            x += diffX;
            y += diffY;
        }

        // Center first object
        if (i == 0)
        {
            centerX = SCREEN_WIDTH/2-(Planet::planets[0]->getX()+x);
            centerY = SCREEN_HEIGHT/2-(Planet::planets[0]->getY()+y);
            Planet::planets[0]->setSpeed(0.0f);
        }

        // Just plan moving, it still has to influence other planets
        Planet::planets[i]->planMoving(x+centerX, y+centerY);
    }

    // Reposition all planets
    for (unsigned int i = 0; i < Planet::planets.size(); i++)
    { Planet::planets[i]->finishMoving(); }

    // Print info
    pspDebugScreenSetXY(0, 0);
    printf("X = add planet, /\\ = reset, [] = demo");
}

void drawPlanets()
{
    for (vector<Planet*>::const_iterator it = Planet::planets.begin();
         it != Planet::planets.end();
         ++it)
    {
        unsigned int color = massToColor((*it)->getMass());

        int intX = (int)(*it)->getX();
        int intY = (int)(*it)->getY();

        drawBox(intX-1, intY,   intX+3, intY+2, color);
        drawBox(intX,   intY-1, intX+2, intY+3, color);
    }
}

// Supported range: [1e0s, 1e6)
int massToColor(float mass)
{
    // Default to white
    if (mass < 1 || mass >= 1e6)
    { return 0xffffffff; }

    // Normalize to 1024
    int color = logf(mass)/logf(10.0f)/6*1024;

    // 256 shades of red+green up
    if (color <= 255)
    { return RGB(255, color, 0); }
    // 256 shades of green+red down
    else if (color <= 511)
    { return RGB(511-color, 255, 0); }
    // 256 shades of green+blue up
    else if (color <= 767)
    { return RGB(0, 255, color-512); }
    // 256 shades of blue+green down
    else
    { return RGB(0, 1023-color, 255); }
}

void createPlanet()
{
    Planet *last = NULL;
    if (Planet::planets.size() > 0)
    { last = Planet::planets.back(); }

    // No planets, or last planet is already finished => start a new one
    if (last == NULL || last->finished())
    { Planet::planets.push_back(new Planet(1e3, SCREEN_WIDTH/4, SCREEN_HEIGHT/4, 0)); }
    // Continue setting previously created planet
    else
    {
        // Change mass
        float growth = 1.2;
        if (pad.Buttons & PSP_CTRL_LTRIGGER && last->getMass() > 1.0f*growth)
        { last->setMass(last->getMass()/growth); }
        else if (pad.Buttons & PSP_CTRL_RTRIGGER && last->getMass() < 1e6f/growth)
        { last->setMass(last->getMass()*growth); }

        // Change position
        float newX = last->getX() + (pad.Lx < 96 || pad.Lx > 160 ? (pad.Lx-128)/16 : 0);
        float newY = last->getY() + (pad.Ly < 96 || pad.Ly > 160 ? (pad.Ly-128)/16 : 0);
        limitToScreen(&newX, &newY);
        last->setX(newX);
        last->setY(newY);

        // Print info
        pspDebugScreenSetXY(0, 0);
        printf("Mass: %.1E (L/R to change)", last->getMass());
    }
}

void directPlanet()
{
    Planet *last = Planet::planets.back();
    float newSpeed = 0.0f, newDirection = 0.0f;
    float newX = last->getX() + sinf(last->getDirection())*last->getSpeed();
    float newY = last->getY() + cosf(last->getDirection())*last->getSpeed();

    // Go back
    if (pad.Buttons & PSP_CTRL_TRIANGLE)
    {
        last->unfinish();
        currentState = addPlanet;
    }
    // Find stable orbit
    if (pad.Buttons & PSP_CTRL_CIRCLE)
    {
        // There are other planets
        if (Planet::planets.size() > 1)
        {
            // Find strongest planet
            Planet *strongestPlanet = NULL; float strongestForce = 0.0f; float strongestDistance = 0.0f;
            for (unsigned int i = 0; i < Planet::planets.size()-1; i++)
            {
                float diffX = Planet::planets[i]->getX() - last->getX();
                float diffY = Planet::planets[i]->getY() - last->getY();
                float distance = sqrtf(SQUARE(diffX) + SQUARE(diffY));

                float force = (Planet::planets[i]->getMass() * last->getMass())
                             /(SQUARE(distance));
                if (force > strongestForce)
                {
                    strongestPlanet = Planet::planets[i];
                    strongestForce = force;
                    strongestDistance = distance;
                }
            }

            // New direction is perpendicular to direction to strongest planet
            newDirection = atan2(strongestPlanet->getX() - last->getX()
                                ,strongestPlanet->getY() - last->getY());
            newDirection += GU_PI/2;

            // Stable orbit is impossible
            if (strongestForce/last->getMass()/2 > strongestDistance)
            {
                newSpeed = 1; // new position is >90° of orbit, all is lost?
                newX = last->getX();
                newY = last->getY();
            }
            // Stable orbit around a planet
            else
            {
                // New speed depends on strongest planet (see dev/stable.txt)
                newSpeed = sqrtf(SQUARE(strongestDistance)-SQUARE(strongestDistance-strongestForce/last->getMass()/2));

                newX = last->getX() + sinf(last->getDirection())*last->getSpeed();
                newY = last->getY() + cosf(last->getDirection())*last->getSpeed();

                // TODO: Add current momentum vector of strongest planet
                /*
                newX += sinf(strongestPlanet->getDirection())*strongestPlanet->getSpeed();
                newY += cosf(strongestPlanet->getDirection())*strongestPlanet->getSpeed();

                newDirection = atan2(newX - last->getX()
                                    ,newY - last->getY());
                newDirection += GU_PI/2;

                newSpeed = sqrtf(SQUARE(newX - last->getX())+SQUARE(newY - last->getY()));
                */
            }
        }
    }
    // Free orbit
    else
    {
        // Calculate vector of momentum and its change
        newX += pad.Lx < 96 || pad.Lx > 160 ? (pad.Lx-128)/16 : 0;
        newY += pad.Ly < 96 || pad.Ly > 160 ? (pad.Ly-128)/16 : 0;
        limitToScreen(&newX, &newY);

        // Extract new speed and direction
        newSpeed = sqrtf(SQUARE(last->getX()-newX) + SQUARE(last->getY()-newY));
        newDirection = atan2(newX - last->getX(), newY - last->getY());
        if (newDirection < 0)
        { newDirection += 2*GU_PI; }
    }

    last->setSpeed(newSpeed);
    last->setDirection(newDirection);

    // Show vector
    drawLine(last->getX(), last->getY(), newX+1, newY+1, massToColor(last->getMass()));

    // Print info
    pspDebugScreenSetXY(0, 0);
    printf("Speed: %.1f, direction: %d deg.\n(O for stable orbit, /\\ to go back)", last->getSpeed(), (int)(last->getDirection()*180.0f/GU_PI));
}

template <class T>
void limitToScreen(T *x, T *y)
{
    if (*x < 0)              { *x = 0; }
    if (*x >= SCREEN_WIDTH)  { *x = SCREEN_WIDTH-1; }
    if (*y < 0)              { *y = 0; }
    if (*y >= SCREEN_HEIGHT) { *y = SCREEN_HEIGHT-1; }
}
